﻿using Microsoft.AspNetCore.Mvc.Formatters;
using Microsoft.Net.Http.Headers;
using System;
using System.IO;
using System.Text;
using System.Threading.Tasks;
using VA.PPMS.IWS.Api.Data;

namespace VA.PPMS.IWS.Api.Formatters
{
    public class FhirInputFormatter : TextInputFormatter
    {
        public FhirInputFormatter()
        {
            SupportedMediaTypes.Add(MediaTypeHeaderValue.Parse("application/fhir+json"));

            SupportedEncodings.Add(Encoding.UTF8);
            SupportedEncodings.Add(Encoding.Unicode);
        }

        protected override bool CanReadType(Type type)
        {
            return type == typeof(string) && base.CanReadType(type);
        }

        public override bool CanRead(InputFormatterContext context)
        {
            if (context == null) throw new ArgumentNullException(nameof(context));

            var request = context.HttpContext.Request;
            var contentType = request.ContentType;

            return string.IsNullOrEmpty(contentType) ||
                   contentType == "text/plain" ||
                   contentType == "application/json+fhir" ||
                   contentType == "application/json";
        }

        public override async Task<InputFormatterResult> ReadRequestBodyAsync(InputFormatterContext context, Encoding encoding)
        {
            if (context == null)
            {
                throw new ArgumentNullException(nameof(context));
            }

            if (encoding == null)
            {
                throw new ArgumentNullException(nameof(encoding));
            }

            try
            {
                var message = new FhirMessage();
                await message.LoadFhirMessage(context.HttpContext.Request, encoding);

                if (message.IsValid) return await InputFormatterResult.SuccessAsync(message.FhirMessagePayload);

                throw new InvalidDataException("The request data could not be processed.");
            }
            catch (Exception)
            {
                return await InputFormatterResult.FailureAsync();
            }
        }
    }
}